<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;
use App\Models\Order;
use App\Models\Review;

class HomeController extends Controller
{
    public function index()
    {
        // المنتجات المميزة
        $featuredProducts = Product::active()
            ->featured()
            ->with('category')
            ->take(6)
            ->get();

        // أشهر المنتجات
        $popularProducts = Product::active()
            ->popular()
            ->with('category')
            ->take(6)
            ->get();

        // الفئات النشطة
        $categories = Category::active()
            ->ordered()
            ->withCount('activeProducts')
            ->take(6)
            ->get();

        // إحصائيات عامة
        $stats = [
            'happy_customers' => Order::where('order_status', Order::STATUS_DELIVERED)->distinct('customer_phone')->count('customer_phone'),
            'daily_orders' => Order::today()->count(),
            'average_delivery_time' => 25, // يمكن حسابها من البيانات الفعلية
            'customer_rating' => Review::avg('rating') ?? 4.8
        ];

        // آخر التقييمات
        $latestReviews = Review::with('user', 'product')
            ->where('rating', '>=', 4)
            ->latest()
            ->take(5)
            ->get();

        return view('home', compact(
            'featuredProducts',
            'popularProducts', 
            'categories',
            'stats',
            'latestReviews'
        ));
    }

    public function about()
    {
        $stats = [
            'establishment_year' => 2018,
            'happy_customers' => Order::where('order_status', Order::STATUS_DELIVERED)->distinct('customer_phone')->count('customer_phone'),
            'average_rating' => Review::avg('rating') ?? 4.8,
            'delivery_coverage' => '24/7'
        ];

        $achievements = [
            'أول مطعم برجر في دمنهور يقدم التوصيل خلال 30 دقيقة',
            'حاصل على شهادة الجودة من وزارة الصحة المصرية', 
            'أكثر من 50,000 عميل راضي عن خدماتنا',
            'متاح 24/7 لخدمتك في أي وقت',
            'فريق عمل مدرب على أعلى مستوى',
            'استخدام أجود أنواع اللحوم والخضار الطازجة'
        ];

        $team = [
            [
                'name' => 'أحمد محمد',
                'position' => 'المدير العام',
                'image' => 'team-1.jpg',
                'experience' => 'خبرة 15 سنة في مجال المطاعم'
            ],
            [
                'name' => 'فاطمة علي', 
                'position' => 'رئيس الطهاة',
                'image' => 'team-2.jpg',
                'experience' => 'متخصصة في الطبخ الغربي والشرقي'
            ],
            [
                'name' => 'محمود حسن',
                'position' => 'مدير خدمة العملاء', 
                'image' => 'team-3.jpg',
                'experience' => 'خبرة 8 سنوات في خدمة العملاء'
            ]
        ];

        return view('about', compact('stats', 'achievements', 'team'));
    }

    public function contact()
    {
        $contactInfo = [
            [
                'type' => 'phone',
                'title' => 'اتصل بنا',
                'value' => '01234567890',
                'description' => 'متاح 24/7',
                'icon' => 'phone'
            ],
            [
                'type' => 'email', 
                'title' => 'راسلنا',
                'value' => 'info@burgeronfire.com',
                'description' => 'سنرد خلال ساعتين',
                'icon' => 'mail'
            ],
            [
                'type' => 'address',
                'title' => 'موقعنا', 
                'value' => 'دمنهور، البحيرة، مصر',
                'description' => 'شارع الجمهورية الرئيسي',
                'icon' => 'map-pin'
            ],
            [
                'type' => 'hours',
                'title' => 'ساعات العمل',
                'value' => '24 ساعة طوال الأسبوع', 
                'description' => 'خدمة توصيل مستمرة',
                'icon' => 'clock'
            ]
        ];

        $faq = [
            [
                'question' => 'كم يستغرق التوصيل؟',
                'answer' => 'نضمن التوصيل خلال 30 دقيقة أو أقل داخل نطاق دمنهور'
            ],
            [
                'question' => 'هل يوجد حد أدنى للطلب؟',
                'answer' => 'الحد الأدنى للطلب 50 جنيه، والتوصيل مجاني للطلبات أكثر من 150 جنيه'
            ],
            [
                'question' => 'ما هي طرق الدفع المتاحة؟',
                'answer' => 'نقبل الدفع نقداً عند التسليم، فودافون كاش، وبطاقات الائتمان'
            ],
            [
                'question' => 'هل يمكن تخصيص الطلب؟',
                'answer' => 'بالطبع! يمكنك إضافة أو حذف أي مكونات حسب رغبتك'
            ]
        ];

        return view('contact', compact('contactInfo', 'faq'));
    }

    public function submitContact(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'email' => 'nullable|email|max:255',
            'subject' => 'nullable|string|max:255',
            'message' => 'required|string|max:1000'
        ]);

        // هنا يمكن إرسال بريد إلكتروني أو حفظ في قاعدة البيانات
        // Mail::to('info@burgeronfire.com')->send(new ContactMessage($request->all()));

        return back()->with('success', 'تم إرسال رسالتك بنجاح. سنتواصل معك قريباً!');
    }

    public function newsletter(Request $request)
    {
        $request->validate([
            'email' => 'required|email|max:255'
        ]);

        // هنا يمكن حفظ البريد في قائمة النشرة الإخبارية
        // Newsletter::create(['email' => $request->email]);

        return back()->with('success', 'تم الاشتراك في النشرة الإخبارية بنجاح!');
    }

    public function search(Request $request)
    {
        $query = $request->get('q');
        
        if (empty($query)) {
            return redirect()->route('menu');
        }

        $products = Product::active()
            ->search($query)
            ->with('category')
            ->paginate(12);

        $categories = Category::active()
            ->where('name', 'like', "%{$query}%")
            ->get();

        return view('search-results', compact('products', 'categories', 'query'));
    }
}
