<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class Order extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'order_number',
        'customer_name',
        'customer_phone',
        'customer_email',
        'delivery_address',
        'delivery_instructions',
        'payment_method',
        'payment_status',
        'order_status',
        'subtotal',
        'delivery_fee',
        'discount_amount',
        'coupon_code',
        'tax_amount',
        'total_amount',
        'delivery_time',
        'special_instructions',
        'driver_id',
        'estimated_delivery_time',
        'delivered_at',
        'cancelled_at',
        'cancellation_reason'
    ];

    protected $casts = [
        'delivery_address' => 'array',
        'subtotal' => 'decimal:2',
        'delivery_fee' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'estimated_delivery_time' => 'datetime',
        'delivered_at' => 'datetime',
        'cancelled_at' => 'datetime'
    ];

    // Order Status Constants
    const STATUS_PENDING = 'pending';
    const STATUS_CONFIRMED = 'confirmed';
    const STATUS_PREPARING = 'preparing';
    const STATUS_READY = 'ready';
    const STATUS_ON_WAY = 'on_way';
    const STATUS_DELIVERED = 'delivered';
    const STATUS_CANCELLED = 'cancelled';

    // Payment Status Constants
    const PAYMENT_PENDING = 'pending';
    const PAYMENT_PAID = 'paid';
    const PAYMENT_FAILED = 'failed';
    const PAYMENT_REFUNDED = 'refunded';

    // Payment Methods
    const PAYMENT_CASH = 'cash';
    const PAYMENT_CARD = 'card';
    const PAYMENT_VODAFONE = 'vodafone_cash';

    // العلاقات
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function items()
    {
        return $this->hasMany(OrderItem::class);
    }

    public function driver()
    {
        return $this->belongsTo(User::class, 'driver_id');
    }

    public function statusHistory()
    {
        return $this->hasMany(OrderStatusHistory::class);
    }

    public function review()
    {
        return $this->hasOne(OrderReview::class);
    }

    // Scopes
    public function scopeToday($query)
    {
        return $query->whereDate('created_at', today());
    }

    public function scopeThisWeek($query)
    {
        return $query->whereBetween('created_at', [
            Carbon::now()->startOfWeek(),
            Carbon::now()->endOfWeek()
        ]);
    }

    public function scopeThisMonth($query)
    {
        return $query->whereMonth('created_at', Carbon::now()->month)
                    ->whereYear('created_at', Carbon::now()->year);
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('order_status', $status);
    }

    public function scopeByPaymentMethod($query, $method)
    {
        return $query->where('payment_method', $method);
    }

    // Helper Methods
    public function generateOrderNumber()
    {
        return 'ORD' . date('Ymd') . str_pad($this->id, 4, '0', STR_PAD_LEFT);
    }

    public function getStatusTextAttribute()
    {
        $statuses = [
            self::STATUS_PENDING => 'في انتظار التأكيد',
            self::STATUS_CONFIRMED => 'تم تأكيد الطلب',
            self::STATUS_PREPARING => 'جاري التحضير',
            self::STATUS_READY => 'جاهز للتوصيل',
            self::STATUS_ON_WAY => 'في الطريق إليك',
            self::STATUS_DELIVERED => 'تم التوصيل',
            self::STATUS_CANCELLED => 'تم الإلغاء'
        ];

        return $statuses[$this->order_status] ?? 'غير محدد';
    }

    public function getPaymentStatusTextAttribute()
    {
        $statuses = [
            self::PAYMENT_PENDING => 'في انتظار الدفع',
            self::PAYMENT_PAID => 'تم الدفع',
            self::PAYMENT_FAILED => 'فشل الدفع',
            self::PAYMENT_REFUNDED => 'تم الاسترداد'
        ];

        return $statuses[$this->payment_status] ?? 'غير محدد';
    }

    public function getPaymentMethodTextAttribute()
    {
        $methods = [
            self::PAYMENT_CASH => 'الدفع عند التسليم',
            self::PAYMENT_CARD => 'بطاقة ائتمان',
            self::PAYMENT_VODAFONE => 'فودافون كاش'
        ];

        return $methods[$this->payment_method] ?? 'غير محدد';
    }

    public function canBeCancelled()
    {
        return in_array($this->order_status, [
            self::STATUS_PENDING,
            self::STATUS_CONFIRMED
        ]);
    }

    public function isDelivered()
    {
        return $this->order_status === self::STATUS_DELIVERED;
    }

    public function isCancelled()
    {
        return $this->order_status === self::STATUS_CANCELLED;
    }

    public function isPaid()
    {
        return $this->payment_status === self::PAYMENT_PAID;
    }

    public function getEstimatedDeliveryText()
    {
        if (!$this->estimated_delivery_time) {
            return 'غير محدد';
        }

        $now = Carbon::now();
        $estimated = Carbon::parse($this->estimated_delivery_time);
        
        if ($estimated->isFuture()) {
            return $estimated->diffForHumans() . ' (' . $estimated->format('H:i') . ')';
        }
        
        return 'متأخر ' . $estimated->diffForHumans();
    }

    public function updateStatus($newStatus, $note = null)
    {
        $oldStatus = $this->order_status;
        $this->order_status = $newStatus;
        
        // Set delivered_at timestamp
        if ($newStatus === self::STATUS_DELIVERED && !$this->delivered_at) {
            $this->delivered_at = now();
        }
        
        // Set cancelled_at timestamp
        if ($newStatus === self::STATUS_CANCELLED && !$this->cancelled_at) {
            $this->cancelled_at = now();
        }
        
        $this->save();
        
        // Add to status history
        $this->statusHistory()->create([
            'old_status' => $oldStatus,
            'new_status' => $newStatus,
            'note' => $note,
            'changed_by' => auth()->id()
        ]);
        
        return true;
    }

    public function calculateTotal()
    {
        $subtotal = $this->items()->sum(\DB::raw('price * quantity'));
        $this->subtotal = $subtotal;
        $this->total_amount = $subtotal + $this->delivery_fee + $this->tax_amount - $this->discount_amount;
        $this->save();
        
        return $this->total_amount;
    }

    public function getFormattedTotalAttribute()
    {
        return number_format($this->total_amount, 0) . ' جنيه';
    }

    public function getItemsCountAttribute()
    {
        return $this->items()->sum('quantity');
    }

    public function hasReview()
    {
        return $this->review()->exists();
    }

    protected static function boot()
    {
        parent::boot();
        
        static::created(function ($order) {
            if (!$order->order_number) {
                $order->order_number = $order->generateOrderNumber();
                $order->save();
            }
        });
    }
}
