<?php

namespace App\Models;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\SoftDeletes;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable, SoftDeletes;

    protected $fillable = [
        'name',
        'email',
        'phone',
        'email_verified_at',
        'password',
        'avatar',
        'date_of_birth',
        'gender',
        'loyalty_points',
        'membership_level',
        'is_active',
        'last_login_at',
        'preferences'
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'date_of_birth' => 'date',
        'is_active' => 'boolean',
        'last_login_at' => 'datetime',
        'preferences' => 'array'
    ];

    // Membership Levels
    const LEVEL_BRONZE = 'برونزي';
    const LEVEL_SILVER = 'فضي';
    const LEVEL_GOLD = 'ذهبي';
    const LEVEL_PLATINUM = 'بلاتيني';

    // العلاقات
    public function orders()
    {
        return $this->hasMany(Order::class);
    }

    public function addresses()
    {
        return $this->hasMany(UserAddress::class);
    }

    public function favorites()
    {
        return $this->hasMany(Favorite::class);
    }

    public function reviews()
    {
        return $this->hasMany(Review::class);
    }

    public function cartItems()
    {
        return $this->hasMany(CartItem::class);
    }

    public function defaultAddress()
    {
        return $this->hasOne(UserAddress::class)->where('is_default', true);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByMembershipLevel($query, $level)
    {
        return $query->where('membership_level', $level);
    }

    public function scopeWithLoyaltyPoints($query, $min = 0)
    {
        return $query->where('loyalty_points', '>=', $min);
    }

    // Helper Methods
    public function getMembershipBenefits()
    {
        $benefits = [
            self::LEVEL_BRONZE => [
                'discount' => 0,
                'points_multiplier' => 1,
                'free_delivery' => false,
                'special_offers' => false,
                'priority_support' => false
            ],
            self::LEVEL_SILVER => [
                'discount' => 5,
                'points_multiplier' => 1.5,
                'free_delivery' => false,
                'special_offers' => true,
                'priority_support' => false
            ],
            self::LEVEL_GOLD => [
                'discount' => 10,
                'points_multiplier' => 2,
                'free_delivery' => true,
                'special_offers' => true,
                'priority_support' => true
            ],
            self::LEVEL_PLATINUM => [
                'discount' => 15,
                'points_multiplier' => 3,
                'free_delivery' => true,
                'special_offers' => true,
                'priority_support' => true
            ]
        ];

        return $benefits[$this->membership_level] ?? $benefits[self::LEVEL_BRONZE];
    }

    public function addLoyaltyPoints($points, $reason = null)
    {
        $benefits = $this->getMembershipBenefits();
        $pointsToAdd = $points * $benefits['points_multiplier'];
        
        $this->loyalty_points += $pointsToAdd;
        $this->updateMembershipLevel();
        $this->save();

        // Log the points addition
        $this->loyaltyPointsHistory()->create([
            'points' => $pointsToAdd,
            'type' => 'earned',
            'reason' => $reason,
            'balance_after' => $this->loyalty_points
        ]);

        return $pointsToAdd;
    }

    public function redeemLoyaltyPoints($points)
    {
        if ($this->loyalty_points < $points) {
            return false;
        }

        $this->loyalty_points -= $points;
        $this->save();

        // Log the points redemption
        $this->loyaltyPointsHistory()->create([
            'points' => -$points,
            'type' => 'redeemed',
            'reason' => 'تم استخدام النقاط للحصول على خصم',
            'balance_after' => $this->loyalty_points
        ]);

        return true;
    }

    public function updateMembershipLevel()
    {
        $newLevel = self::LEVEL_BRONZE;
        
        if ($this->loyalty_points >= 5000) {
            $newLevel = self::LEVEL_PLATINUM;
        } elseif ($this->loyalty_points >= 2000) {
            $newLevel = self::LEVEL_GOLD;
        } elseif ($this->loyalty_points >= 500) {
            $newLevel = self::LEVEL_SILVER;
        }

        if ($this->membership_level !== $newLevel) {
            $oldLevel = $this->membership_level;
            $this->membership_level = $newLevel;
            
            // Log membership level change
            // You can add a notification or email here
        }
    }

    public function getAvatarUrlAttribute()
    {
        if ($this->avatar) {
            return asset('storage/avatars/' . $this->avatar);
        }
        
        // Generate initials avatar
        $initials = strtoupper(substr($this->name, 0, 1));
        if (strpos($this->name, ' ') !== false) {
            $nameParts = explode(' ', $this->name);
            $initials = strtoupper(substr($nameParts[0], 0, 1) . substr($nameParts[1], 0, 1));
        }
        
        return "https://ui-avatars.com/api/?name={$initials}&background=f97316&color=fff&size=100";
    }

    public function getFormattedPhoneAttribute()
    {
        if (!$this->phone) return '';
        
        // Format Egyptian phone number
        $phone = preg_replace('/[^0-9]/', '', $this->phone);
        if (strlen($phone) == 11 && substr($phone, 0, 2) == '01') {
            return substr($phone, 0, 3) . ' ' . substr($phone, 3, 3) . ' ' . substr($phone, 6);
        }
        
        return $this->phone;
    }

    public function getTotalOrdersAttribute()
    {
        return $this->orders()->count();
    }

    public function getTotalSpentAttribute()
    {
        return $this->orders()
                   ->where('order_status', Order::STATUS_DELIVERED)
                   ->sum('total_amount');
    }

    public function getLastOrderAttribute()
    {
        return $this->orders()->latest()->first();
    }

    public function getFavoriteProductsCountAttribute()
    {
        return $this->favorites()->count();
    }

    public function getAddressesCountAttribute()
    {
        return $this->addresses()->count();
    }

    public function getCartItemsCountAttribute()
    {
        return $this->cartItems()->sum('quantity');
    }

    public function isEligibleForFreeDelivery()
    {
        return $this->getMembershipBenefits()['free_delivery'];
    }

    public function getDiscountPercentage()
    {
        return $this->getMembershipBenefits()['discount'];
    }

    public function hasSpecialOffers()
    {
        return $this->getMembershipBenefits()['special_offers'];
    }

    public function hasPrioritySupport()
    {
        return $this->getMembershipBenefits()['priority_support'];
    }

    public function getPointsToNextLevel()
    {
        $currentPoints = $this->loyalty_points;
        
        if ($currentPoints < 500) {
            return 500 - $currentPoints;
        } elseif ($currentPoints < 2000) {
            return 2000 - $currentPoints;
        } elseif ($currentPoints < 5000) {
            return 5000 - $currentPoints;
        }
        
        return 0; // Already at highest level
    }

    public function getNextMembershipLevel()
    {
        $currentLevel = $this->membership_level;
        
        switch ($currentLevel) {
            case self::LEVEL_BRONZE:
                return self::LEVEL_SILVER;
            case self::LEVEL_SILVER:
                return self::LEVEL_GOLD;
            case self::LEVEL_GOLD:
                return self::LEVEL_PLATINUM;
            default:
                return null; // Already at highest level
        }
    }

    public function loyaltyPointsHistory()
    {
        return $this->hasMany(LoyaltyPointHistory::class);
    }

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($user) {
            if (!$user->membership_level) {
                $user->membership_level = self::LEVEL_BRONZE;
            }
            if (!$user->loyalty_points) {
                $user->loyalty_points = 0;
            }
            if (!$user->is_active) {
                $user->is_active = true;
            }
        });
    }
}
