<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\ProductController;
use App\Http\Controllers\CartController;
use App\Http\Controllers\OrderController;
use App\Http\Controllers\UserController;
use App\Http\Controllers\AuthController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
*/

// الصفحة الرئيسية
Route::get('/', [HomeController::class, 'index'])->name('home');

// الصفحات الثابتة
Route::get('/about', [HomeController::class, 'about'])->name('about');
Route::get('/contact', [HomeController::class, 'contact'])->name('contact');
Route::post('/contact', [HomeController::class, 'submitContact'])->name('contact.submit');
Route::post('/newsletter', [HomeController::class, 'newsletter'])->name('newsletter.subscribe');

// البحث العام
Route::get('/search', [HomeController::class, 'search'])->name('search');

// المنتجات والقائمة
Route::prefix('menu')->name('products.')->group(function () {
    Route::get('/', [ProductController::class, 'index'])->name('index');
    Route::get('/featured', [ProductController::class, 'featured'])->name('featured');
    Route::get('/popular', [ProductController::class, 'popular'])->name('popular');
    Route::get('/offers', [ProductController::class, 'offers'])->name('offers');
    Route::get('/search', [ProductController::class, 'search'])->name('search');
    Route::get('/autocomplete', [ProductController::class, 'autocomplete'])->name('autocomplete');
    Route::get('/category/{category}', [ProductController::class, 'getByCategory'])->name('category');
    Route::get('/{product}', [ProductController::class, 'show'])->name('show');
    
    // AJAX Routes
    Route::get('/{product}/quick-view', [ProductController::class, 'quickView'])->name('quick-view');
    Route::post('/{product}/favorite', [ProductController::class, 'addToFavorites'])->name('favorite');
    Route::get('/{product}/customizations', [ProductController::class, 'getCustomizationOptions'])->name('customizations');
});

// سلة التسوق
Route::prefix('cart')->name('cart.')->group(function () {
    Route::get('/', [CartController::class, 'index'])->name('index');
    Route::post('/add', [CartController::class, 'add'])->name('add');
    Route::patch('/update/{item}', [CartController::class, 'update'])->name('update');
    Route::delete('/remove/{item}', [CartController::class, 'remove'])->name('remove');
    Route::delete('/clear', [CartController::class, 'clear'])->name('clear');
    Route::post('/apply-coupon', [CartController::class, 'applyCoupon'])->name('apply-coupon');
    Route::delete('/remove-coupon', [CartController::class, 'removeCoupon'])->name('remove-coupon');
    Route::get('/count', [CartController::class, 'getCount'])->name('count');
});

// المصادقة
Route::prefix('auth')->name('auth.')->group(function () {
    Route::get('/login', [AuthController::class, 'showLogin'])->name('login');
    Route::post('/login', [AuthController::class, 'login'])->name('login.submit');
    Route::get('/register', [AuthController::class, 'showRegister'])->name('register');
    Route::post('/register', [AuthController::class, 'register'])->name('register.submit');
    Route::post('/logout', [AuthController::class, 'logout'])->name('logout')->middleware('auth');
    
    // نسيان كلمة المرور
    Route::get('/forgot-password', [AuthController::class, 'showForgotPassword'])->name('forgot-password');
    Route::post('/forgot-password', [AuthController::class, 'forgotPassword'])->name('forgot-password.submit');
    Route::get('/reset-password/{token}', [AuthController::class, 'showResetPassword'])->name('reset-password');
    Route::post('/reset-password', [AuthController::class, 'resetPassword'])->name('reset-password.submit');
});

// الطلبات
Route::middleware('auth')->prefix('orders')->name('orders.')->group(function () {
    Route::get('/', [OrderController::class, 'index'])->name('index');
    Route::get('/checkout', [OrderController::class, 'showCheckout'])->name('checkout');
    Route::post('/checkout', [OrderController::class, 'processCheckout'])->name('checkout.submit');
    Route::get('/{order}', [OrderController::class, 'show'])->name('show');
    Route::get('/{order}/track', [OrderController::class, 'track'])->name('track');
    Route::post('/{order}/cancel', [OrderController::class, 'cancel'])->name('cancel');
    Route::post('/{order}/review', [OrderController::class, 'submitReview'])->name('review');
    Route::get('/{order}/receipt', [OrderController::class, 'receipt'])->name('receipt');
    Route::get('/{order}/reorder', [OrderController::class, 'reorder'])->name('reorder');
});

// الملف الشخصي
Route::middleware('auth')->prefix('profile')->name('profile.')->group(function () {
    Route::get('/', [UserController::class, 'profile'])->name('index');
    Route::patch('/', [UserController::class, 'updateProfile'])->name('update');
    Route::post('/avatar', [UserController::class, 'updateAvatar'])->name('avatar');
    
    // العناوين
    Route::get('/addresses', [UserController::class, 'addresses'])->name('addresses');
    Route::post('/addresses', [UserController::class, 'addAddress'])->name('addresses.add');
    Route::patch('/addresses/{address}', [UserController::class, 'updateAddress'])->name('addresses.update');
    Route::delete('/addresses/{address}', [UserController::class, 'deleteAddress'])->name('addresses.delete');
    Route::post('/addresses/{address}/default', [UserController::class, 'setDefaultAddress'])->name('addresses.default');
    
    // المفضلة
    Route::get('/favorites', [UserController::class, 'favorites'])->name('favorites');
    Route::delete('/favorites/{favorite}', [UserController::class, 'removeFavorite'])->name('favorites.remove');
    
    // نقاط الولاء
    Route::get('/loyalty', [UserController::class, 'loyalty'])->name('loyalty');
    Route::post('/loyalty/redeem', [UserController::class, 'redeemPoints'])->name('loyalty.redeem');
    
    // الإعدادات
    Route::get('/settings', [UserController::class, 'settings'])->name('settings');
    Route::patch('/settings', [UserController::class, 'updateSettings'])->name('settings.update');
    Route::patch('/password', [UserController::class, 'updatePassword'])->name('password.update');
});

// API Routes للتطبيقات الخارجية
Route::prefix('api')->name('api.')->group(function () {
    // معلومات المطعم
    Route::get('/restaurant-info', function () {
        return response()->json([
            'name' => config('app.name'),
            'phone' => env('RESTAURANT_PHONE'),
            'email' => env('RESTAURANT_EMAIL'),
            'address' => env('RESTAURANT_ADDRESS'),
            'delivery_fee' => env('DELIVERY_FEE'),
            'free_delivery_threshold' => env('FREE_DELIVERY_THRESHOLD'),
            'delivery_time' => [
                'min' => env('DELIVERY_TIME_MIN'),
                'max' => env('DELIVERY_TIME_MAX')
            ],
            'working_hours' => '24/7',
            'social_media' => [
                'facebook' => env('FACEBOOK_URL'),
                'instagram' => env('INSTAGRAM_URL'), 
                'twitter' => env('TWITTER_URL')
            ]
        ]);
    });
    
    // الفئات
    Route::get('/categories', function () {
        return response()->json(
            \App\Models\Category::active()
                ->withCount('activeProducts')
                ->ordered()
                ->get()
        );
    });
    
    // المنتجات
    Route::get('/products', function (Request $request) {
        $query = \App\Models\Product::active()->with('category');
        
        if ($request->has('category')) {
            $query->byCategory($request->category);
        }
        
        if ($request->has('featured')) {
            $query->featured();
        }
        
        if ($request->has('popular')) {
            $query->popular();
        }
        
        return response()->json(
            $query->paginate($request->get('per_page', 12))
        );
    });
    
    // منتج محدد
    Route::get('/products/{product}', function (\App\Models\Product $product) {
        if (!$product->is_active) {
            return response()->json(['error' => 'المنتج غير متاح'], 404);
        }
        
        return response()->json($product->load('category'));
    });
});

// Webhooks للدفع الإلكتروني
Route::post('/webhooks/stripe', [OrderController::class, 'stripeWebhook'])->name('webhooks.stripe');
Route::post('/webhooks/paypal', [OrderController::class, 'paypalWebhook'])->name('webhooks.paypal');

// Sitemap
Route::get('/sitemap.xml', function () {
    $products = \App\Models\Product::active()->get();
    $categories = \App\Models\Category::active()->get();
    
    return response()->view('sitemap', compact('products', 'categories'))
                   ->header('Content-Type', 'application/xml');
});

// Robots.txt
Route::get('/robots.txt', function () {
    $content = "User-agent: *\n";
    $content .= "Allow: /\n";
    $content .= "Sitemap: " . url('/sitemap.xml') . "\n";
    
    return response($content, 200, ['Content-Type' => 'text/plain']);
});

// تتبع الطلبات للضيوف (بدون تسجيل دخول)
Route::get('/track-order', [OrderController::class, 'showGuestTracking'])->name('guest.track');
Route::post('/track-order', [OrderController::class, 'guestTracking'])->name('guest.track.submit');

// صفحة الخطأ المخصصة
Route::fallback(function () {
    return view('errors.404');
});
